<?php
// Direct database connection test to diagnose the 500 error

echo "<h2>Database Connection Debug</h2>";

// Try to load Laravel's environment
try {
    require_once '/home/alakhbarpressapp/public_html/last/application/vendor/autoload.php';
    
    // Load environment variables
    $dotenv = Dotenv\Dotenv::createImmutable('/home/alakhbarpressapp/public_html/last/application');
    $dotenv->load();
    
    echo "<p>✅ Laravel autoloader loaded successfully</p>";
    
    // Get database credentials from .env
    $host = $_ENV['DB_HOST'] ?? 'localhost';
    $database = $_ENV['DB_DATABASE'] ?? '';
    $username = $_ENV['DB_USERNAME'] ?? '';
    $password = $_ENV['DB_PASSWORD'] ?? '';
    
    echo "<p><strong>Database Config:</strong></p>";
    echo "<ul>";
    echo "<li>Host: " . htmlspecialchars($host) . "</li>";
    echo "<li>Database: " . htmlspecialchars($database) . "</li>";
    echo "<li>Username: " . htmlspecialchars($username) . "</li>";
    echo "<li>Password: " . (empty($password) ? 'Empty' : '[Hidden]') . "</li>";
    echo "</ul>";
    
} catch (Exception $e) {
    echo "<p>❌ Error loading Laravel: " . htmlspecialchars($e->getMessage()) . "</p>";
    
    // Fallback: try to read .env file directly
    $envFile = '/home/alakhbarpressapp/public_html/last/application/.env';
    if (file_exists($envFile)) {
        echo "<p>📄 Reading .env file directly...</p>";
        $envContent = file_get_contents($envFile);
        
        // Extract database credentials
        preg_match('/DB_HOST=(.*)/', $envContent, $hostMatch);
        preg_match('/DB_DATABASE=(.*)/', $envContent, $dbMatch);
        preg_match('/DB_USERNAME=(.*)/', $envContent, $userMatch);
        preg_match('/DB_PASSWORD=(.*)/', $envContent, $passMatch);
        
        $host = isset($hostMatch[1]) ? trim($hostMatch[1]) : 'localhost';
        $database = isset($dbMatch[1]) ? trim($dbMatch[1]) : '';
        $username = isset($userMatch[1]) ? trim($userMatch[1]) : '';
        $password = isset($passMatch[1]) ? trim($passMatch[1]) : '';
        
        echo "<p><strong>Database Config from .env:</strong></p>";
        echo "<ul>";
        echo "<li>Host: " . htmlspecialchars($host) . "</li>";
        echo "<li>Database: " . htmlspecialchars($database) . "</li>";
        echo "<li>Username: " . htmlspecialchars($username) . "</li>";
        echo "<li>Password: " . (empty($password) ? 'Empty' : '[Hidden]') . "</li>";
        echo "</ul>";
    } else {
        echo "<p>❌ .env file not found at: $envFile</p>";
        exit;
    }
}

// Test database connection
echo "<h3>Testing Database Connection...</h3>";

try {
    $dsn = "mysql:host=$host;dbname=$database;charset=utf8mb4";
    $pdo = new PDO($dsn, $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
    
    echo "<p>✅ Database connection successful!</p>";
    
    // Check if tables exist
    echo "<h3>Checking Tables...</h3>";
    
    $tables = ['notices', 'categories'];
    foreach ($tables as $table) {
        try {
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM $table");
            $result = $stmt->fetch();
            echo "<p>✅ Table '$table' exists with {$result['count']} records</p>";
        } catch (Exception $e) {
            echo "<p>❌ Table '$table' error: " . htmlspecialchars($e->getMessage()) . "</p>";
        }
    }
    
    // Test the actual query from NoticeController
    echo "<h3>Testing NoticeController Query...</h3>";
    
    try {
        $sql = "SELECT notices.*, categories.name_ara as category_name_ara, categories.name_fr as category_name_fr 
                FROM notices 
                JOIN categories ON categories.id = notices.category_id 
                ORDER BY notices.order DESC";
        
        $stmt = $pdo->query($sql);
        $results = $stmt->fetchAll();
        
        echo "<p>✅ NoticeController query successful! Found " . count($results) . " records</p>";
        
        if (count($results) > 0) {
            echo "<p><strong>Sample record:</strong></p>";
            echo "<pre>" . htmlspecialchars(json_encode($results[0], JSON_PRETTY_PRINT)) . "</pre>";
        }
        
    } catch (Exception $e) {
        echo "<p>❌ NoticeController query failed: " . htmlspecialchars($e->getMessage()) . "</p>";
    }
    
} catch (Exception $e) {
    echo "<p>❌ Database connection failed: " . htmlspecialchars($e->getMessage()) . "</p>";
}

echo "<h3>PHP Information</h3>";
echo "<p>PHP Version: " . phpversion() . "</p>";
echo "<p>PDO Available: " . (extension_loaded('pdo') ? 'Yes' : 'No') . "</p>";
echo "<p>PDO MySQL Available: " . (extension_loaded('pdo_mysql') ? 'Yes' : 'No') . "</p>";
?>
